import { defineComponent, ref, computed, mergeProps, unref, useSSRContext } from 'vue';
import { ssrRenderAttrs, ssrInterpolate, ssrRenderList, ssrRenderClass, ssrIncludeBooleanAttr, ssrLooseContain, ssrLooseEqual, ssrRenderStyle, ssrRenderAttr } from 'vue/server-renderer';

const _sfc_main = /* @__PURE__ */ defineComponent({
  __name: "projects",
  __ssrInlineRender: true,
  setup(__props) {
    const activeTab = ref("internal");
    const showCreateProjectModal = ref(false);
    const showCreateTaskModal = ref(false);
    ref(false);
    ref(false);
    const projectStatusFilter = ref("");
    const projectPriorityFilter = ref("");
    const externalProjectStatusFilter = ref("");
    const taskStatusFilter = ref("");
    const taskAssigneeFilter = ref("");
    const timelineView = ref("week");
    const tabs = [
      { id: "internal", name: "Internal Projects" },
      { id: "external", name: "External Projects" },
      { id: "tasks", name: "Tasks" },
      { id: "team", name: "Team" }
    ];
    const projectData = ref({
      projects: [
        // Internal Projects
        {
          id: "1",
          name: "Website Redesign",
          description: "Complete redesign of the company website with modern UI/UX",
          status: "active",
          priority: "high",
          startDate: /* @__PURE__ */ new Date("2024-01-01"),
          endDate: /* @__PURE__ */ new Date("2024-03-31"),
          budget: 25e3,
          progress: 65,
          teamMembers: ["1", "2", "3"],
          clientId: "client-1",
          type: "internal"
        },
        {
          id: "2",
          name: "Mobile App Development",
          description: "Development of a new mobile application for iOS and Android",
          status: "planning",
          priority: "urgent",
          startDate: /* @__PURE__ */ new Date("2024-02-01"),
          endDate: /* @__PURE__ */ new Date("2024-06-30"),
          budget: 5e4,
          progress: 15,
          teamMembers: ["2", "4", "5"],
          clientId: "client-2",
          type: "internal"
        },
        {
          id: "3",
          name: "Marketing Campaign",
          description: "Launch of a comprehensive marketing campaign for Q2",
          status: "completed",
          priority: "medium",
          startDate: /* @__PURE__ */ new Date("2023-12-01"),
          endDate: /* @__PURE__ */ new Date("2024-01-31"),
          budget: 15e3,
          progress: 100,
          teamMembers: ["1", "3"],
          clientId: "client-3",
          type: "internal"
        },
        // External Projects (from Zoho)
        {
          id: "4",
          name: "Zoho CRM Integration",
          description: "Integration project from Zoho CRM system",
          status: "in_progress",
          priority: "high",
          startDate: /* @__PURE__ */ new Date("2024-01-15"),
          endDate: /* @__PURE__ */ new Date("2024-04-15"),
          budget: 3e4,
          progress: 40,
          teamMembers: ["1", "4"],
          clientId: "zoho-client-1",
          type: "external",
          zohoId: "ZP-001"
        },
        {
          id: "5",
          name: "Zoho Books Migration",
          description: "Migration project from Zoho Books",
          status: "not_started",
          priority: "medium",
          startDate: /* @__PURE__ */ new Date("2024-03-01"),
          endDate: /* @__PURE__ */ new Date("2024-05-31"),
          budget: 2e4,
          progress: 0,
          teamMembers: ["2", "3"],
          clientId: "zoho-client-2",
          type: "external",
          zohoId: "ZP-002"
        },
        {
          id: "6",
          name: "Zoho Analytics Dashboard",
          description: "Analytics dashboard project from Zoho",
          status: "completed",
          priority: "low",
          startDate: /* @__PURE__ */ new Date("2023-11-01"),
          endDate: /* @__PURE__ */ new Date("2023-12-31"),
          budget: 12e3,
          progress: 100,
          teamMembers: ["3", "5"],
          clientId: "zoho-client-3",
          type: "external",
          zohoId: "ZP-003"
        }
      ],
      tasks: [
        {
          id: "1",
          projectId: "1",
          title: "Design Homepage Layout",
          description: "Create wireframes and mockups for the new homepage",
          status: "in_progress",
          priority: "high",
          assigneeId: "1",
          dueDate: /* @__PURE__ */ new Date("2024-02-15"),
          createdAt: /* @__PURE__ */ new Date("2024-01-15"),
          updatedAt: /* @__PURE__ */ new Date("2024-01-20")
        },
        {
          id: "2",
          projectId: "1",
          title: "Implement Responsive Design",
          description: "Make the website responsive for all device sizes",
          status: "todo",
          priority: "medium",
          assigneeId: "2",
          dueDate: /* @__PURE__ */ new Date("2024-02-28"),
          createdAt: /* @__PURE__ */ new Date("2024-01-16"),
          updatedAt: /* @__PURE__ */ new Date("2024-01-16")
        },
        {
          id: "3",
          projectId: "2",
          title: "Setup Development Environment",
          description: "Configure development tools and environment for mobile app",
          status: "completed",
          priority: "high",
          assigneeId: "4",
          dueDate: /* @__PURE__ */ new Date("2024-01-31"),
          createdAt: /* @__PURE__ */ new Date("2024-01-10"),
          updatedAt: /* @__PURE__ */ new Date("2024-01-25")
        }
      ],
      teamMembers: [
        {
          id: "1",
          name: "John Smith",
          email: "john@example.com",
          role: "UI/UX Designer",
          avatar: "",
          isActive: true
        },
        {
          id: "2",
          name: "Sarah Johnson",
          email: "sarah@example.com",
          role: "Frontend Developer",
          avatar: "",
          isActive: true
        },
        {
          id: "3",
          name: "Mike Wilson",
          email: "mike@example.com",
          role: "Backend Developer",
          avatar: "",
          isActive: true
        },
        {
          id: "4",
          name: "Emily Davis",
          email: "emily@example.com",
          role: "Mobile Developer",
          avatar: "",
          isActive: true
        },
        {
          id: "5",
          name: "David Brown",
          email: "david@example.com",
          role: "Project Manager",
          avatar: "",
          isActive: false
        }
      ]
    });
    const newProject = ref({
      name: "",
      description: "",
      priority: "",
      startDate: "",
      endDate: "",
      budget: 0
    });
    const newTask = ref({
      title: "",
      description: "",
      projectId: "",
      assigneeId: "",
      priority: "",
      dueDate: ""
    });
    computed(() => {
      let projects = projectData.value.projects;
      if (projectStatusFilter.value) {
        projects = projects.filter((p) => p.status === projectStatusFilter.value);
      }
      if (projectPriorityFilter.value) {
        projects = projects.filter(
          (p) => p.priority === projectPriorityFilter.value
        );
      }
      return projects;
    });
    const filteredTasks = computed(() => {
      let tasks = projectData.value.tasks;
      if (taskStatusFilter.value) {
        tasks = tasks.filter((t) => t.status === taskStatusFilter.value);
      }
      if (taskAssigneeFilter.value) {
        tasks = tasks.filter((t) => t.assigneeId === taskAssigneeFilter.value);
      }
      return tasks;
    });
    const getInternalProjects = () => {
      return projectData.value.projects.filter(
        (project) => project.type === "internal"
      );
    };
    const getExternalProjects = () => {
      return projectData.value.projects.filter(
        (project) => project.type === "external"
      );
    };
    const filteredInternalProjects = computed(() => {
      let projects = getInternalProjects();
      if (projectStatusFilter.value) {
        projects = projects.filter((p) => p.status === projectStatusFilter.value);
      }
      if (projectPriorityFilter.value) {
        projects = projects.filter(
          (p) => p.priority === projectPriorityFilter.value
        );
      }
      return projects;
    });
    const filteredExternalProjects = computed(() => {
      let projects = getExternalProjects();
      if (externalProjectStatusFilter.value) {
        projects = projects.filter(
          (p) => p.status === externalProjectStatusFilter.value
        );
      }
      return projects;
    });
    const formatDate = (date) => {
      return new Intl.DateTimeFormat("en-US", {
        year: "numeric",
        month: "short",
        day: "numeric"
      }).format(date);
    };
    const isOverdue = (dueDate) => {
      if (!dueDate) return false;
      return /* @__PURE__ */ new Date() > dueDate;
    };
    const getProjectStatusBadgeClass = (status) => {
      switch (status) {
        case "active":
          return "badge-success";
        case "planning":
          return "badge-info";
        case "on_hold":
          return "badge-warning";
        case "completed":
          return "badge-success";
        case "cancelled":
          return "badge-error";
        default:
          return "badge-info";
      }
    };
    const getTaskStatusBadgeClass = (status) => {
      switch (status) {
        case "todo":
          return "badge-info";
        case "in_progress":
          return "badge-warning";
        case "review":
          return "badge-warning";
        case "completed":
          return "badge-success";
        default:
          return "badge-info";
      }
    };
    const getPriorityBadgeClass = (priority) => {
      switch (priority) {
        case "low":
          return "badge-info";
        case "medium":
          return "badge-warning";
        case "high":
          return "badge-error";
        case "urgent":
          return "badge-error";
        default:
          return "badge-info";
      }
    };
    const getProjectName = (projectId) => {
      const project = projectData.value.projects.find((p) => p.id === projectId);
      return project ? project.name : "Unknown Project";
    };
    const getTeamMemberName = (memberId) => {
      const member = projectData.value.teamMembers.find((m) => m.id === memberId);
      return member ? member.name : "Unknown Member";
    };
    const getActiveProjects = () => {
      return projectData.value.projects.filter((p) => p.status === "active").length;
    };
    const getCompletedTasks = () => {
      return projectData.value.tasks.filter((t) => t.status === "completed").length;
    };
    return (_ctx, _push, _parent, _attrs) => {
      _push(`<div${ssrRenderAttrs(mergeProps({ class: "projects-module" }, _attrs))}><div class="content-body"><div class="kpi-grid mb-8"><div class="kpi-card"><div class="kpi-value">${ssrInterpolate(getInternalProjects().length)}</div><div class="kpi-label">Internal Projects</div><div class="kpi-change positive">+2 this month</div></div><div class="kpi-card"><div class="kpi-value">${ssrInterpolate(getExternalProjects().length)}</div><div class="kpi-label">External Projects</div><div class="kpi-change">Synced from Zoho</div></div><div class="kpi-card"><div class="kpi-value">${ssrInterpolate(getActiveProjects())}</div><div class="kpi-label">Active Projects</div><div class="kpi-change positive">+1 this week</div></div><div class="kpi-card"><div class="kpi-value">${ssrInterpolate(getCompletedTasks())}</div><div class="kpi-label">Completed Tasks</div><div class="kpi-change positive">+12 this week</div></div></div><div class="tabs-container card mb-8"><div class="tabs-header"><!--[-->`);
      ssrRenderList(tabs, (tab) => {
        _push(`<button class="${ssrRenderClass([{ active: unref(activeTab) === tab.id }, "tab-button"])}">${ssrInterpolate(tab.name)}</button>`);
      });
      _push(`<!--]--></div><div class="tabs-content">`);
      if (unref(activeTab) === "internal") {
        _push(`<div class="tab-pane"><div class="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4 mb-4"><h3 class="text-lg font-semibold">Internal Projects</h3><div class="flex flex-row flex-wrap items-center gap-2 filters-row"><select class="form-select"><option value=""${ssrIncludeBooleanAttr(Array.isArray(unref(projectStatusFilter)) ? ssrLooseContain(unref(projectStatusFilter), "") : ssrLooseEqual(unref(projectStatusFilter), "")) ? " selected" : ""}>All Status</option><option value="planning"${ssrIncludeBooleanAttr(Array.isArray(unref(projectStatusFilter)) ? ssrLooseContain(unref(projectStatusFilter), "planning") : ssrLooseEqual(unref(projectStatusFilter), "planning")) ? " selected" : ""}>Planning</option><option value="active"${ssrIncludeBooleanAttr(Array.isArray(unref(projectStatusFilter)) ? ssrLooseContain(unref(projectStatusFilter), "active") : ssrLooseEqual(unref(projectStatusFilter), "active")) ? " selected" : ""}>Active</option><option value="on_hold"${ssrIncludeBooleanAttr(Array.isArray(unref(projectStatusFilter)) ? ssrLooseContain(unref(projectStatusFilter), "on_hold") : ssrLooseEqual(unref(projectStatusFilter), "on_hold")) ? " selected" : ""}>On Hold</option><option value="completed"${ssrIncludeBooleanAttr(Array.isArray(unref(projectStatusFilter)) ? ssrLooseContain(unref(projectStatusFilter), "completed") : ssrLooseEqual(unref(projectStatusFilter), "completed")) ? " selected" : ""}>Completed</option><option value="cancelled"${ssrIncludeBooleanAttr(Array.isArray(unref(projectStatusFilter)) ? ssrLooseContain(unref(projectStatusFilter), "cancelled") : ssrLooseEqual(unref(projectStatusFilter), "cancelled")) ? " selected" : ""}>Cancelled</option></select><select class="form-select"><option value=""${ssrIncludeBooleanAttr(Array.isArray(unref(projectPriorityFilter)) ? ssrLooseContain(unref(projectPriorityFilter), "") : ssrLooseEqual(unref(projectPriorityFilter), "")) ? " selected" : ""}>All Priority</option><option value="low"${ssrIncludeBooleanAttr(Array.isArray(unref(projectPriorityFilter)) ? ssrLooseContain(unref(projectPriorityFilter), "low") : ssrLooseEqual(unref(projectPriorityFilter), "low")) ? " selected" : ""}>Low</option><option value="medium"${ssrIncludeBooleanAttr(Array.isArray(unref(projectPriorityFilter)) ? ssrLooseContain(unref(projectPriorityFilter), "medium") : ssrLooseEqual(unref(projectPriorityFilter), "medium")) ? " selected" : ""}>Medium</option><option value="high"${ssrIncludeBooleanAttr(Array.isArray(unref(projectPriorityFilter)) ? ssrLooseContain(unref(projectPriorityFilter), "high") : ssrLooseEqual(unref(projectPriorityFilter), "high")) ? " selected" : ""}>High</option><option value="urgent"${ssrIncludeBooleanAttr(Array.isArray(unref(projectPriorityFilter)) ? ssrLooseContain(unref(projectPriorityFilter), "urgent") : ssrLooseEqual(unref(projectPriorityFilter), "urgent")) ? " selected" : ""}>Urgent</option></select></div></div><div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-2 gap-4 lg:gap-6 projects-grid"><!--[-->`);
        ssrRenderList(unref(filteredInternalProjects), (project) => {
          _push(`<div class="card"><div class="card-body"><div class="flex flex-col sm:flex-row sm:items-start sm:justify-between gap-2 mb-4"><div class="flex flex-wrap items-center gap-2"><span class="badge badge-internal">Internal</span><span class="${ssrRenderClass([getProjectStatusBadgeClass(project.status), "badge"])}">${ssrInterpolate(project.status)}</span><span class="${ssrRenderClass([getPriorityBadgeClass(project.priority), "badge"])}">${ssrInterpolate(project.priority)}</span></div></div><h4 class="font-semibold text-lg mb-2">${ssrInterpolate(project.name)}</h4><p class="text-gray-600 text-sm mb-4">${ssrInterpolate(project.description)}</p><div class="mb-4"><div class="flex items-center justify-between text-sm text-gray-600 mb-1"><span>Progress</span><span>${ssrInterpolate(project.progress)}%</span></div><div class="w-full bg-gray-200 rounded-full h-2"><div class="bg-blue-600 h-2 rounded-full" style="${ssrRenderStyle({ width: `${project.progress}%` })}"></div></div></div><div class="flex items-center justify-between text-sm text-gray-500 mb-4"><span>${ssrInterpolate(project.teamMembers.length)} members</span>`);
          if (project.budget) {
            _push(`<span>\u20AC${ssrInterpolate(project.budget.toLocaleString())}</span>`);
          } else {
            _push(`<!---->`);
          }
          _push(`</div><div class="flex flex-col sm:flex-row items-stretch sm:items-center gap-2"><button class="btn btn-primary flex-1" title="View Project"><svg class="btn-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"></path><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"></path></svg></button><button class="btn btn-outline"><svg class="btn-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"></path></svg></button></div></div></div>`);
        });
        _push(`<!--]--></div></div>`);
      } else {
        _push(`<!---->`);
      }
      if (unref(activeTab) === "external") {
        _push(`<div class="tab-pane"><div class="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4 mb-4"><h3 class="text-lg font-semibold">External Projects (Zoho)</h3><div class="flex flex-row flex-wrap items-center gap-2 filters-row"><select class="form-select"><option value=""${ssrIncludeBooleanAttr(Array.isArray(unref(externalProjectStatusFilter)) ? ssrLooseContain(unref(externalProjectStatusFilter), "") : ssrLooseEqual(unref(externalProjectStatusFilter), "")) ? " selected" : ""}>All Status</option><option value="not_started"${ssrIncludeBooleanAttr(Array.isArray(unref(externalProjectStatusFilter)) ? ssrLooseContain(unref(externalProjectStatusFilter), "not_started") : ssrLooseEqual(unref(externalProjectStatusFilter), "not_started")) ? " selected" : ""}>Not Started</option><option value="in_progress"${ssrIncludeBooleanAttr(Array.isArray(unref(externalProjectStatusFilter)) ? ssrLooseContain(unref(externalProjectStatusFilter), "in_progress") : ssrLooseEqual(unref(externalProjectStatusFilter), "in_progress")) ? " selected" : ""}>In Progress</option><option value="on_hold"${ssrIncludeBooleanAttr(Array.isArray(unref(externalProjectStatusFilter)) ? ssrLooseContain(unref(externalProjectStatusFilter), "on_hold") : ssrLooseEqual(unref(externalProjectStatusFilter), "on_hold")) ? " selected" : ""}>On Hold</option><option value="completed"${ssrIncludeBooleanAttr(Array.isArray(unref(externalProjectStatusFilter)) ? ssrLooseContain(unref(externalProjectStatusFilter), "completed") : ssrLooseEqual(unref(externalProjectStatusFilter), "completed")) ? " selected" : ""}>Completed</option><option value="cancelled"${ssrIncludeBooleanAttr(Array.isArray(unref(externalProjectStatusFilter)) ? ssrLooseContain(unref(externalProjectStatusFilter), "cancelled") : ssrLooseEqual(unref(externalProjectStatusFilter), "cancelled")) ? " selected" : ""}>Cancelled</option></select><button class="btn btn-outline"><svg class="btn-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"></path></svg> Refresh </button></div></div><div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-2 gap-4 lg:gap-6 projects-grid"><!--[-->`);
        ssrRenderList(unref(filteredExternalProjects), (project) => {
          _push(`<div class="card"><div class="card-body"><div class="flex flex-col sm:flex-row sm:items-start sm:justify-between gap-2 mb-4"><div class="flex flex-wrap items-center gap-2"><span class="badge badge-external">External</span><span class="${ssrRenderClass([getProjectStatusBadgeClass(project.status), "badge"])}">${ssrInterpolate(project.status)}</span><span class="${ssrRenderClass([getPriorityBadgeClass(project.priority), "badge"])}">${ssrInterpolate(project.priority)}</span></div></div><h4 class="font-semibold text-lg mb-2">${ssrInterpolate(project.name)}</h4><p class="text-gray-600 text-sm mb-4">${ssrInterpolate(project.description)}</p><div class="mb-4"><div class="flex justify-between text-sm mb-1"><span>Progress</span><span>${ssrInterpolate(project.progress)}%</span></div><div class="w-full bg-gray-200 rounded-full h-2"><div class="bg-green-600 h-2 rounded-full" style="${ssrRenderStyle({ width: `${project.progress}%` })}"></div></div></div><div class="flex items-center justify-between text-sm text-gray-500 mb-4"><span>${ssrInterpolate(project.teamMembers.length)} members</span>`);
          if (project.budget) {
            _push(`<span>\u20AC${ssrInterpolate(project.budget.toLocaleString())}</span>`);
          } else {
            _push(`<!---->`);
          }
          _push(`</div><div class="flex flex-col sm:flex-row items-stretch sm:items-center gap-2"><button class="btn btn-primary flex-1" title="View Project"><svg class="btn-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"></path><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"></path></svg></button><button class="btn btn-outline"><svg class="btn-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"></path></svg></button></div></div></div>`);
        });
        _push(`<!--]--></div></div>`);
      } else {
        _push(`<!---->`);
      }
      if (unref(activeTab) === "tasks") {
        _push(`<div class="tab-pane"><div class="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4 mb-4"><h3 class="text-lg font-semibold">Recent Tasks</h3><div class="flex flex-row flex-wrap items-center gap-2 filters-row"><select class="form-select"><option value=""${ssrIncludeBooleanAttr(Array.isArray(unref(taskStatusFilter)) ? ssrLooseContain(unref(taskStatusFilter), "") : ssrLooseEqual(unref(taskStatusFilter), "")) ? " selected" : ""}>All Status</option><option value="todo"${ssrIncludeBooleanAttr(Array.isArray(unref(taskStatusFilter)) ? ssrLooseContain(unref(taskStatusFilter), "todo") : ssrLooseEqual(unref(taskStatusFilter), "todo")) ? " selected" : ""}>To Do</option><option value="in_progress"${ssrIncludeBooleanAttr(Array.isArray(unref(taskStatusFilter)) ? ssrLooseContain(unref(taskStatusFilter), "in_progress") : ssrLooseEqual(unref(taskStatusFilter), "in_progress")) ? " selected" : ""}>In Progress</option><option value="review"${ssrIncludeBooleanAttr(Array.isArray(unref(taskStatusFilter)) ? ssrLooseContain(unref(taskStatusFilter), "review") : ssrLooseEqual(unref(taskStatusFilter), "review")) ? " selected" : ""}>Review</option><option value="completed"${ssrIncludeBooleanAttr(Array.isArray(unref(taskStatusFilter)) ? ssrLooseContain(unref(taskStatusFilter), "completed") : ssrLooseEqual(unref(taskStatusFilter), "completed")) ? " selected" : ""}>Completed</option></select><select class="form-select"><option value=""${ssrIncludeBooleanAttr(Array.isArray(unref(taskAssigneeFilter)) ? ssrLooseContain(unref(taskAssigneeFilter), "") : ssrLooseEqual(unref(taskAssigneeFilter), "")) ? " selected" : ""}>All Assignees</option><!--[-->`);
        ssrRenderList(unref(projectData).teamMembers, (member) => {
          _push(`<option${ssrRenderAttr("value", member.id)}${ssrIncludeBooleanAttr(Array.isArray(unref(taskAssigneeFilter)) ? ssrLooseContain(unref(taskAssigneeFilter), member.id) : ssrLooseEqual(unref(taskAssigneeFilter), member.id)) ? " selected" : ""}>${ssrInterpolate(member.name)}</option>`);
        });
        _push(`<!--]--></select></div></div><div class="overflow-x-auto"><table class="table"><thead><tr><th>Task</th><th>Project</th><th>Assignee</th><th>Priority</th><th>Status</th><th>Due Date</th><th>Actions</th></tr></thead><tbody><!--[-->`);
        ssrRenderList(unref(filteredTasks), (task) => {
          _push(`<tr><td><div><div class="font-medium">${ssrInterpolate(task.title)}</div><div class="text-sm text-gray-500">${ssrInterpolate(task.description)}</div></div></td><td>${ssrInterpolate(getProjectName(task.projectId))}</td><td>${ssrInterpolate(getTeamMemberName(task.assigneeId))}</td><td><span class="${ssrRenderClass([getPriorityBadgeClass(task.priority), "badge"])}">${ssrInterpolate(task.priority)}</span></td><td><span class="${ssrRenderClass([getTaskStatusBadgeClass(task.status), "badge"])}">${ssrInterpolate(task.status)}</span></td><td><span class="${ssrRenderClass(
            isOverdue(task.dueDate) ? "text-red-600" : "text-gray-600"
          )}">${ssrInterpolate(task.dueDate ? formatDate(task.dueDate) : "No due date")}</span></td><td><div class="flex items-center gap-1"><button class="btn btn-sm btn-outline" title="View Task"><svg class="btn-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"></path><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"></path></svg></button><button class="btn btn-sm btn-outline" title="Edit Task"><svg class="btn-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"></path></svg></button><button class="btn btn-sm btn-primary" title="Update Status"><svg class="btn-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg></button></div></td></tr>`);
        });
        _push(`<!--]--></tbody></table></div></div>`);
      } else {
        _push(`<!---->`);
      }
      if (unref(activeTab) === "team") {
        _push(`<div class="tab-pane"><div class="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4 mb-4"><h3 class="text-lg font-semibold">Recent Team Members</h3><button class="btn btn-primary" title="Add Member"><svg class="btn-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6"></path></svg></button></div><div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 lg:gap-6"><!--[-->`);
        ssrRenderList(unref(projectData).teamMembers, (member) => {
          _push(`<div class="card"><div class="card-body text-center"><div class="w-16 h-16 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-4"><span class="text-2xl font-bold text-blue-600">${ssrInterpolate(member.name.charAt(0))}</span></div><h4 class="font-semibold text-lg mb-1">${ssrInterpolate(member.name)}</h4><p class="text-gray-600 text-sm mb-2">${ssrInterpolate(member.email)}</p><p class="text-gray-500 text-sm mb-4">${ssrInterpolate(member.role)}</p><div class="flex items-center justify-center gap-2"><span class="${ssrRenderClass([member.isActive ? "badge-success" : "badge-error", "badge"])}">${ssrInterpolate(member.isActive ? "Active" : "Inactive")}</span></div><div class="flex items-center justify-center gap-2 mt-4"><button class="btn btn-sm btn-outline" title="Edit Member"><svg class="btn-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"></path></svg></button><button class="btn btn-sm btn-error" title="Remove Member"><svg class="btn-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"></path></svg></button></div></div></div>`);
        });
        _push(`<!--]--></div></div>`);
      } else {
        _push(`<!---->`);
      }
      if (unref(activeTab) === "timeline") {
        _push(`<div><div class="flex items-center justify-between mb-4"><h3 class="text-lg font-semibold">Project Timeline</h3><div class="flex gap-2"><select class="form-select"><option value="week"${ssrIncludeBooleanAttr(Array.isArray(unref(timelineView)) ? ssrLooseContain(unref(timelineView), "week") : ssrLooseEqual(unref(timelineView), "week")) ? " selected" : ""}>This Week</option><option value="month"${ssrIncludeBooleanAttr(Array.isArray(unref(timelineView)) ? ssrLooseContain(unref(timelineView), "month") : ssrLooseEqual(unref(timelineView), "month")) ? " selected" : ""}>This Month</option><option value="quarter"${ssrIncludeBooleanAttr(Array.isArray(unref(timelineView)) ? ssrLooseContain(unref(timelineView), "quarter") : ssrLooseEqual(unref(timelineView), "quarter")) ? " selected" : ""}>This Quarter</option></select></div></div><div class="space-y-4"><!--[-->`);
        ssrRenderList(unref(projectData).projects, (project) => {
          _push(`<div class="card"><div class="card-body"><div class="flex items-center justify-between mb-4"><h4 class="font-semibold text-lg">${ssrInterpolate(project.name)}</h4><span class="${ssrRenderClass([getProjectStatusBadgeClass(project.status), "badge"])}">${ssrInterpolate(project.status)}</span></div><div class="flex items-center gap-4 text-sm text-gray-600"><span>Start: ${ssrInterpolate(formatDate(project.startDate))}</span>`);
          if (project.endDate) {
            _push(`<span>End: ${ssrInterpolate(formatDate(project.endDate))}</span>`);
          } else {
            _push(`<!---->`);
          }
          _push(`<span>Progress: ${ssrInterpolate(project.progress)}%</span></div><div class="mt-3"><div class="w-full bg-gray-200 rounded-full h-2"><div class="bg-blue-600 h-2 rounded-full" style="${ssrRenderStyle({ width: `${project.progress}%` })}"></div></div></div></div></div>`);
        });
        _push(`<!--]--></div></div>`);
      } else {
        _push(`<!---->`);
      }
      _push(`</div></div></div>`);
      if (unref(showCreateProjectModal)) {
        _push(`<div class="modal-overlay"><div class="modal"><div class="modal-header"><h3 class="text-lg font-semibold">Create New Project</h3><button class="text-gray-400 hover:text-gray-600"><svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path></svg></button></div><div class="modal-body"><form><div class="form-group"><label class="form-label">Project Name</label><input${ssrRenderAttr("value", unref(newProject).name)} type="text" class="form-input" required></div><div class="form-group"><label class="form-label">Description</label><textarea class="form-textarea" required>${ssrInterpolate(unref(newProject).description)}</textarea></div><div class="form-group"><label class="form-label">Priority</label><select class="form-select" required><option value=""${ssrIncludeBooleanAttr(Array.isArray(unref(newProject).priority) ? ssrLooseContain(unref(newProject).priority, "") : ssrLooseEqual(unref(newProject).priority, "")) ? " selected" : ""}>Select Priority</option><option value="low"${ssrIncludeBooleanAttr(Array.isArray(unref(newProject).priority) ? ssrLooseContain(unref(newProject).priority, "low") : ssrLooseEqual(unref(newProject).priority, "low")) ? " selected" : ""}>Low</option><option value="medium"${ssrIncludeBooleanAttr(Array.isArray(unref(newProject).priority) ? ssrLooseContain(unref(newProject).priority, "medium") : ssrLooseEqual(unref(newProject).priority, "medium")) ? " selected" : ""}>Medium</option><option value="high"${ssrIncludeBooleanAttr(Array.isArray(unref(newProject).priority) ? ssrLooseContain(unref(newProject).priority, "high") : ssrLooseEqual(unref(newProject).priority, "high")) ? " selected" : ""}>High</option><option value="urgent"${ssrIncludeBooleanAttr(Array.isArray(unref(newProject).priority) ? ssrLooseContain(unref(newProject).priority, "urgent") : ssrLooseEqual(unref(newProject).priority, "urgent")) ? " selected" : ""}>Urgent</option></select></div><div class="form-group"><label class="form-label">Start Date</label><input${ssrRenderAttr("value", unref(newProject).startDate)} type="date" class="form-input" required></div><div class="form-group"><label class="form-label">End Date</label><input${ssrRenderAttr("value", unref(newProject).endDate)} type="date" class="form-input"></div><div class="form-group"><label class="form-label">Budget (\u20AC)</label><input${ssrRenderAttr("value", unref(newProject).budget)} type="number" class="form-input" min="0"></div></form></div><div class="modal-footer"><button class="btn btn-outline" title="Cancel"><svg class="btn-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path></svg></button><button class="btn btn-primary" title="Create Project"><svg class="btn-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6"></path></svg></button></div></div></div>`);
      } else {
        _push(`<!---->`);
      }
      if (unref(showCreateTaskModal)) {
        _push(`<div class="modal-overlay"><div class="modal"><div class="modal-header"><h3 class="text-lg font-semibold">Create New Task</h3><button class="text-gray-400 hover:text-gray-600"><svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path></svg></button></div><div class="modal-body"><form><div class="form-group"><label class="form-label">Task Title</label><input${ssrRenderAttr("value", unref(newTask).title)} type="text" class="form-input" required></div><div class="form-group"><label class="form-label">Description</label><textarea class="form-textarea" required>${ssrInterpolate(unref(newTask).description)}</textarea></div><div class="form-group"><label class="form-label">Project</label><select class="form-select" required><option value=""${ssrIncludeBooleanAttr(Array.isArray(unref(newTask).projectId) ? ssrLooseContain(unref(newTask).projectId, "") : ssrLooseEqual(unref(newTask).projectId, "")) ? " selected" : ""}>Select Project</option><!--[-->`);
        ssrRenderList(unref(projectData).projects, (project) => {
          _push(`<option${ssrRenderAttr("value", project.id)}${ssrIncludeBooleanAttr(Array.isArray(unref(newTask).projectId) ? ssrLooseContain(unref(newTask).projectId, project.id) : ssrLooseEqual(unref(newTask).projectId, project.id)) ? " selected" : ""}>${ssrInterpolate(project.name)}</option>`);
        });
        _push(`<!--]--></select></div><div class="form-group"><label class="form-label">Assignee</label><select class="form-select" required><option value=""${ssrIncludeBooleanAttr(Array.isArray(unref(newTask).assigneeId) ? ssrLooseContain(unref(newTask).assigneeId, "") : ssrLooseEqual(unref(newTask).assigneeId, "")) ? " selected" : ""}>Select Assignee</option><!--[-->`);
        ssrRenderList(unref(projectData).teamMembers, (member) => {
          _push(`<option${ssrRenderAttr("value", member.id)}${ssrIncludeBooleanAttr(Array.isArray(unref(newTask).assigneeId) ? ssrLooseContain(unref(newTask).assigneeId, member.id) : ssrLooseEqual(unref(newTask).assigneeId, member.id)) ? " selected" : ""}>${ssrInterpolate(member.name)}</option>`);
        });
        _push(`<!--]--></select></div><div class="form-group"><label class="form-label">Priority</label><select class="form-select" required><option value=""${ssrIncludeBooleanAttr(Array.isArray(unref(newTask).priority) ? ssrLooseContain(unref(newTask).priority, "") : ssrLooseEqual(unref(newTask).priority, "")) ? " selected" : ""}>Select Priority</option><option value="low"${ssrIncludeBooleanAttr(Array.isArray(unref(newTask).priority) ? ssrLooseContain(unref(newTask).priority, "low") : ssrLooseEqual(unref(newTask).priority, "low")) ? " selected" : ""}>Low</option><option value="medium"${ssrIncludeBooleanAttr(Array.isArray(unref(newTask).priority) ? ssrLooseContain(unref(newTask).priority, "medium") : ssrLooseEqual(unref(newTask).priority, "medium")) ? " selected" : ""}>Medium</option><option value="high"${ssrIncludeBooleanAttr(Array.isArray(unref(newTask).priority) ? ssrLooseContain(unref(newTask).priority, "high") : ssrLooseEqual(unref(newTask).priority, "high")) ? " selected" : ""}>High</option><option value="urgent"${ssrIncludeBooleanAttr(Array.isArray(unref(newTask).priority) ? ssrLooseContain(unref(newTask).priority, "urgent") : ssrLooseEqual(unref(newTask).priority, "urgent")) ? " selected" : ""}>Urgent</option></select></div><div class="form-group"><label class="form-label">Due Date</label><input${ssrRenderAttr("value", unref(newTask).dueDate)} type="date" class="form-input"></div></form></div><div class="modal-footer"><button class="btn btn-outline" title="Cancel"><svg class="btn-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path></svg></button><button class="btn btn-primary" title="Create Task"><svg class="btn-icon" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 6v6m0 0v6m0-6h6m-6 0H6"></path></svg></button></div></div></div>`);
      } else {
        _push(`<!---->`);
      }
      _push(`</div>`);
    };
  }
});
const _sfc_setup = _sfc_main.setup;
_sfc_main.setup = (props, ctx) => {
  const ssrContext = useSSRContext();
  (ssrContext.modules || (ssrContext.modules = /* @__PURE__ */ new Set())).add("pages/c/[slug]/projects.vue");
  return _sfc_setup ? _sfc_setup(props, ctx) : void 0;
};

export { _sfc_main as default };
//# sourceMappingURL=projects-CKFXIOzg.mjs.map
