import jwt from 'jsonwebtoken'
import bcrypt from 'bcryptjs'

export default defineEventHandler(async (event) => {
  try {
    const body = await readBody(event)
    const { email, password } = body

    if (!email || !password) {
      throw createError({
        statusCode: 400,
        statusMessage: 'Email and password are required'
      })
    }

    // In a real application, you would query your database here
    // For now, we'll use mock data
    const mockUser = {
      id: '1',
      email: 'owner@sidashboard.com',
      password: '$2a$10$cWGjQWVM5XWTjTUMdL0tMOvAqDyVQVq0.xcIMOFm1wSiSAe20qezO', // admin123
      name: 'Owner User',
      role: 'owner',
      isActive: true
    }

    // Verify password
    const isValidPassword = await bcrypt.compare(password, mockUser.password)
    if (!isValidPassword) {
      throw createError({
        statusCode: 401,
        statusMessage: 'Invalid credentials'
      })
    }

    // Generate JWT token
    const config = useRuntimeConfig()
    const token = jwt.sign(
      { 
        userId: mockUser.id, 
        email: mockUser.email, 
        role: mockUser.role 
      },
      config.jwtSecret,
      { expiresIn: '24h' }
    )

    // Set cookie
    setCookie(event, 'auth-token', token, {
      httpOnly: true,
      secure: process.env.NODE_ENV === 'production',
      sameSite: 'strict',
      maxAge: 24 * 60 * 60 // 24 hours
    })

    setCookie(event, 'user', JSON.stringify({
      id: mockUser.id,
      email: mockUser.email,
      name: mockUser.name,
      role: mockUser.role
    }), {
      httpOnly: false,
      secure: process.env.NODE_ENV === 'production',
      sameSite: 'strict',
      maxAge: 24 * 60 * 60
    })

    return {
      success: true,
      data: {
        user: {
          id: mockUser.id,
          email: mockUser.email,
          name: mockUser.name,
          role: mockUser.role
        }
      }
    }
  } catch (error) {
    throw createError({
      statusCode: error.statusCode || 500,
      statusMessage: error.statusMessage || 'Login failed'
    })
  }
})
